<?php
// runner.php
declare(strict_types=1);

require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ERROR | E_PARSE);
date_default_timezone_set('Etc/GMT+0');
set_time_limit(0);

$time_start = microtime(true);

// test=1 -> envía REAL (modo prueba)
// sim=1  -> NO envía (solo simula)
$testMode = isset($_GET['test']) && $_GET['test'] == '1';
$simMode  = isset($_GET['sim']) && $_GET['sim'] == '1';

// por defecto: si no mandas nada, es productivo (envía real)
$sendToRC = !$simMode;

$logFile = __DIR__ . '/logs/runner.log';

// ========== LOG: se SOBREESCRIBE cada ejecución ==========
file_put_contents(
    $logFile,
    "=== RUNNER RC↔PROTRACK ===\n" .
        gmdate('Y-m-d H:i:s') . " UTC\n" .
        "MODE=" . ($simMode ? "SIM" : ($testMode ? "TEST" : "PROD")) . "\n\n"
);
function logline(string $s): void
{
    global $logFile;
    file_put_contents($logFile, $s . "\n", FILE_APPEND);
}

// ========== HELPERS ==========
function http_get_json(string $url): array
{
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_CONNECTTIMEOUT => 20,
        CURLOPT_TIMEOUT => 60,
    ]);
    $resp = curl_exec($ch);
    $err = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) throw new Exception("cURL error: $err");
    $j = json_decode($resp, true);
    if (!is_array($j)) throw new Exception("JSON inválido (HTTP $code): " . substr($resp, 0, 200));
    return $j;
}

function azimuth_to_course($azimuth): string
{
    $a = (float)$azimuth;
    if ($a >= 0 && $a < 22.5) return 'N';
    if ($a >= 22.5 && $a < 67.5) return 'NE';
    if ($a >= 67.5 && $a < 112.5) return 'E';
    if ($a >= 112.5 && $a < 157.5) return 'SE';
    if ($a >= 157.5 && $a < 202.5) return 'S';
    if ($a >= 202.5 && $a < 247.5) return 'SO';
    if ($a >= 247.5 && $a < 292.5) return 'O';
    if ($a >= 292.5 && $a < 337.5) return 'NO';
    if ($a >= 337.5 && $a <= 360) return 'N';
    return 'Desconocido';
}

function ts_to_utc_iso($timestamp): string
{
    $ts = (int)$timestamp;
    $dt = new DateTime("@$ts");
    $dt->setTimezone(new DateTimeZone("UTC"));
    return $dt->format('Y-m-d\TH:i:s');
}

// ========== PROTRACK ==========
function protrack_access_token(string $account, string $password): string
{
    $time = time();
    $signature = md5(md5($password) . $time);
    $url_token = "https://api.protrack365.com/api/authorization?time=$time&account=" . rawurlencode($account) . "&signature=$signature";
    $data = http_get_json($url_token);

    if (!isset($data['record']['access_token'])) {
        throw new Exception("No se obtuvo access_token Protrack para account=$account");
    }
    return (string)$data['record']['access_token'];
}

function protrack_device_list(string $access_token): array
{
    $url_list = "http://api.protrack365.com/api/device/list?access_token=" . rawurlencode($access_token);
    $list = http_get_json($url_list);
    return $list['record'] ?? [];
}

function protrack_track_one(string $access_token, string $imei): array
{
    $url = "http://api.protrack365.com/api/track?access_token=" . rawurlencode($access_token) . "&imeis=" . rawurlencode($imei);
    $t = http_get_json($url);
    return $t['record'][0] ?? [];
}

// ========== RC SOAP ==========
function rc_token(string $rc_userId, string $rc_password): string
{
    $options = [
        'location' => 'http://gps.rcontrol.com.mx/Tracking/wcf/RCService.svc',
        'uri'      => 'http://tempuri.org/',
        'trace'    => true
    ];
    $client = new SoapClient('http://gps.rcontrol.com.mx/Tracking/wcf/RCService.svc?wsdl', $options);

    $params = ['userId' => $rc_userId, 'password' => $rc_password];
    $result = $client->__soapCall('GetUserToken', [$params]);

    $token = $result->GetUserTokenResult->token ?? null;
    if (!$token) throw new Exception("No se obtuvo token RC para userId=$rc_userId");
    return (string)$token;
}

function rc_send_event(string $tokenRC, array $events, bool $sendToRC): array
{
    if (!$sendToRC) {
        return ["SIM" => true, "msg" => "Simulación: NO enviado a RC"];
    }

    $options = [
        'location' => 'http://gps.rcontrol.com.mx/Tracking/wcf/RCService.svc',
        'uri'      => 'http://tempuri.org/',
        'trace'    => true
    ];
    $client = new SoapClient('http://gps.rcontrol.com.mx/Tracking/wcf/RCService.svc?wsdl', $options);

    $objcar = [
        "token"  => $tokenRC,
        "events" => $events
    ];

    $res = $client->__soapCall('GPSAssetTracking', [$objcar]);
    // normalizamos a array simple
    return json_decode(json_encode($res), true) ?: ["raw" => $res];
}

// ========== CARGA CLIENTES ==========
$clientId = isset($_GET['client_id']) ? (int)$_GET['client_id'] : 0;

if ($clientId > 0) {
    $st = pdo()->prepare("SELECT * FROM rc_clients WHERE id=? LIMIT 1");
    $st->execute([$clientId]);
    $clients = $st->fetchAll();
} else {
    $clients = pdo()->query("SELECT * FROM rc_clients WHERE is_active=1 ORDER BY id ASC")->fetchAll();
}

if (!$clients) {
    logline("Sin clientes para ejecutar (client_id=$clientId, activos=1).");
    json_out(["ok" => true, "msg" => "Sin clientes", "dry_run" => $dryRun]);
}

// ========== EJECUCIÓN ==========
$out = [
    "ok" => true,
    "mode" => ($simMode ? "SIM" : ($testMode ? "TEST" : "PROD")),

    "clientes" => [],
];

foreach ($clients as $c) {
    $t0 = microtime(true);

    $clientLabel = "ID={$c['id']} SEDE={$c['sede']} CUSTOMER={$c['customer_id']} {$c['customer_name']}";
    logline("---- CLIENTE: $clientLabel ----");

    $sentCount = 0;
    $detail = [];

    try {
        $tokenRC = rc_token($c['rc_userId'], $c['rc_password']);
        logline("RC token OK (len=" . strlen($tokenRC) . ")");

        $pt_token = protrack_access_token($c['protrack_account'], $c['protrack_password']);
        logline("Protrack token OK (len=" . strlen($pt_token) . ")");

        $devices = protrack_device_list($pt_token);
        logline("Unidades listadas: " . count($devices));

        foreach ($devices as $unidad) {
            $imei   = (string)($unidad["imei"] ?? '');
            $uname  = (string)($unidad["devicename"] ?? '');
            $placas = (string)($unidad["platenumber"] ?? '');

            if ($imei === '') continue;

            $u0 = microtime(true);

            $trk = protrack_track_one($pt_token, $imei);

            $latitude  = $trk["latitude"] ?? 0;
            $longitude = $trk["longitude"] ?? 0;
            $speed     = $trk["speed"] ?? 0;
            $azimuth   = $trk["course"] ?? 0;
            $ignicion = (int)($trk["accstatus"] ?? 0);
            if ($ignicion !== 0 && $ignicion !== 1) $ignicion = 0; // RC solo tolera 0/1
            $timestamp = $trk["gpstime"] ?? time();

            $course = azimuth_to_course($azimuth);
            $utc_datetime = ts_to_utc_iso((int)$timestamp);

            $customer = ["id" => (string)$c["customer_id"], "name" => (string)$c["customer_name"]];

            $events = [
                "Event" => [
                    "altitude" => 0,
                    "asset" => $placas,
                    "battery" => 80,
                    "code" => 0,
                    "course" => $course,
                    "customer" => $customer,
                    "date" => $utc_datetime,
                    "ignition" => $ignicion,
                    "latitude" => $latitude,
                    "longitude" => $longitude,
                    "odometer" => 0,
                    "serialNumber" => $imei,
                    "speed" => $speed,
                    "shipment" => 0
                ]
            ];

            $resp = rc_send_event($tokenRC, $events, $sendToRC);
            $uTime = microtime(true) - $u0;

            $sentCount++;

            // detalle para JSON + log
            $payload = ["token" => "***", "events" => $events]; // oculto token en salida
            $detail[] = [
                "unidad" => $uname,
                "imei" => $imei,
                "payload" => $payload,
                "respuesta_rc" => $resp,
                "tiempo_unidad_seg" => round($uTime, 4)
            ];

            logline("  * {$uname} ({$imei}) OK | t=" . round($uTime, 4) . "s");
            // Si quieres TODO el payload/resp al log, descomenta:
            logline("    payload=" . json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
            logline("    resp=" . json_encode($resp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
        }

        $tClient = microtime(true) - $t0;
        logline("RESUMEN CLIENTE: cliente={$c['customer_name']} unidades_enviadas={$sentCount} tiempo_cliente=" . round($tClient, 4) . "s\n");

        $out["clientes"][] = [
            "cliente" => $clientLabel,
            "unidades_enviadas" => $sentCount,
            "detalle" => $detail,
            "tiempo_cliente_seg" => round($tClient, 4)
        ];
    } catch (Throwable $e) {
        $tClient = microtime(true) - $t0;
        logline("ERROR CLIENTE: " . $e->getMessage());
        logline("tiempo_cliente=" . round($tClient, 4) . "s\n");

        $out["clientes"][] = [
            "cliente" => $clientLabel,
            "error" => $e->getMessage(),
            "unidades_enviadas" => $sentCount,
            "detalle" => $detail,
            "tiempo_cliente_seg" => round($tClient, 4)
        ];
    }
}

$time_end = microtime(true);
$total = $time_end - $time_start;
$out["tiempo_ejecucion_seg"] = round($total, 4);

logline("=== FIN RUNNER === tiempo_total=" . round($total, 4) . "s");

echo json_encode($out, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
