<?php
/*******************************************************
 * Dashboard Units - Múltiples minimapas con etiqueta flotante
 * DB: dashboard_db (nube), Tabla: units
 * - Light/Dark
 * - Múltiples mapas Leaflet
 * - Ordena columnas con persistencia
 * - Export CSV/XLSX
 * - Auto-refresh 10s + contador
 * - Ping a dashboard_data.php cada 30s
 *******************************************************/
header('Content-Type: text/html; charset=utf-8');
date_default_timezone_set('America/Mexico_City');
error_reporting(E_ERROR | E_PARSE);

/* =======================================================
   ✅ CONEXIÓN MYSQL (Nube)
   ======================================================= */
$DB_HOST = "db5018846436.hosting-data.io";
$DB_USER = "dbu974452";
$DB_PASS = "testera32.";
$DB_NAME = "dbs14878639";

$mysqli = new mysqli($DB_HOST, $DB_USER, $DB_PASS, $DB_NAME);
if ($mysqli->connect_error) {
    http_response_code(500);
    echo "<pre>ERROR de conexión MySQL: " . htmlspecialchars($mysqli->connect_error) . "</pre>";
    exit;
}
$mysqli->set_charset("utf8mb4");

/* =======================================================
   ✅ UTILIDADES
   ======================================================= */
function is_unix_ts($v) {
    if (!is_numeric($v)) return false;
    $v = (int)$v;
    return ($v >= 946684800 && $v <= 4102444800); // 2000..2100
}
function parse_ts_mixed($value) {
    if ($value === null || $value === '') return null;
    if (is_unix_ts($value)) return (int)$value;
    $t = strtotime((string)$value);
    return ($t !== false) ? $t : null;
}
function format_interval_hm($seconds) {
    if ($seconds === null || $seconds < 0) return "—";
    $d = intdiv($seconds, 86400);
    $seconds %= 86400;
    $h = intdiv($seconds, 3600);
    $seconds %= 3600;
    $m = intdiv($seconds, 60);
    $parts = [];
    if ($d > 0) $parts[] = "$d D";
    if ($h > 0 || $d > 0) $parts[] = "$h Hrs";
    $parts[] = "$m Min";
    return implode(" ", $parts);
}
function esc($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

/* =======================================================
   ✅ CONSULTA
   ======================================================= */
$sql = "SELECT id, unit_id, unit_name, lat, lon, coordenadas, geohash, velocidad, estado,
               ubicacion, geocercas, bloqueomotor, tiempo_sin_reportar, tiempo_ignicion,
               item_time, ignicion, created_at
        FROM units
        ORDER BY unit_name ASC, unit_id ASC";
$res = $mysqli->query($sql);

$rows = [];
if ($res) {
    while ($r = $res->fetch_assoc()) $rows[] = $r;
    $res->free();
}
$now = time();

/* =======================================================
   ✅ FUNCIÓN RENDER FILAS
   ======================================================= */
function render_rows($rows, $now) {
    ob_start();

    if (empty($rows)) {
        echo '<tr><td colspan="12" class="text-center text-secondary py-5">No hay registros.</td></tr>';
        return ob_get_clean();
    }

    foreach ($rows as $r) {
        $unit = trim($r['unit_name']) !== '' ? $r['unit_name'] : (string)$r['unit_id'];
        $item_ts = parse_ts_mixed($r['item_time']);
        $fecha = $item_ts ? date('Y-m-d', $item_ts) : '—';
        $hora  = $item_ts ? date('H:i', $item_ts) : '—';

        // Coordenadas
        $coordStr = '';
        if (!empty($r['coordenadas'])) {
            $coordStr = trim((string)$r['coordenadas']);
        } elseif ($r['lat'] !== null && $r['lon'] !== null) {
            $coordStr = number_format((float)$r['lat'], 6) . "," . number_format((float)$r['lon'], 6);
        }
        $coordDisp = $coordStr !== '' ? esc($coordStr) : '—';

        // Velocidad
        $vel = is_numeric($r['velocidad']) ? (0 + $r['velocidad']) : 0;

        // Estado
        $estadoTxt = trim((string)$r['estado']);
        if ($estadoTxt === '') $estadoTxt = ($vel > 0) ? 'En tránsito' : 'Detenido';
        $estadoBadge = ($vel > 0)
            ? '<span class="badge" style="background:#22c55e;">En tránsito</span>'
            : '<span class="badge" style="background:#ef4444;">Detenido</span>';

        // Geocerca
        $geo = trim((string)$r['geocercas']);
        $geoBadge = ($geo !== '')
            ? '<span class="badge rounded-pill" style="background:#0ea5e9;">' . esc($geo) . '</span>'
            : '<span class="badge rounded-pill" style="background:#475569;">Sin geocerca</span>';

        // Ubicación
        $ubic = trim((string)$r['ubicacion']);
        $ubic = $ubic !== '' ? esc($ubic) : '—';

		// Paro motor
      	//$paro = (string)$r['bloqueomotor']; 
      	//$paro = ($paro === '1' || strtolower($paro) === 'true') ? 'Sí' : 'No';
		$paro = ((int)$r['bloqueomotor'] === 1) ? 'Sí' : 'No';
      //echo "<td class='small-mono paro-js' data-id='".esc($r['unit_id'])."'>...</td>";




        // Tiempo sin reporte
        $sinRepSec = $item_ts ? max(0, $now - $item_ts) : null;
        $sinRep = format_interval_hm($sinRepSec);

        // Ignición
        $ign = strtoupper(trim((string)$r['ignicion']));
        $tiIgnTs = parse_ts_mixed($r['tiempo_ignicion']);
        if (($ign === 'ENCENDIDA' || $ign === 'ON') && $tiIgnTs !== null) {
            $ignView = format_interval_hm(max(0, $now - $tiIgnTs));
            $ignCell = '<span class="status-dot dot-on"></span><span class="small-mono">' . $ignView . '</span>';
        } elseif ($ign === 'APAGADA' || $ign === 'OFF') {
            $ignCell = '<span class="status-dot dot-off"></span><strong>OFF</strong>';
        } else {
            $ignCell = '<span class="status-dot dot-unk"></span>—';
        }

        // Atributos HTML data-* (para mapas y filtro)
        $dataAttrs =
            ' data-unit="' . esc($unit) . '"' .
            ' data-id="' . esc($r['unit_id']) . '"' .
            ' data-geo="' . esc($geo) . '"' .
            ' data-estado="' . esc($estadoTxt) . '"' .
            ' data-ubic="' . esc($ubic) . '"' .
            ' data-sinrep-sec="' . ($sinRepSec !== null ? $sinRepSec : '-1') . '"' .
            ' data-coord="' . esc($coordStr) . '"' .
            ' data-ign="' . esc($ign) . '"';

        echo "<tr{$dataAttrs}>";
        echo "<td class='fw-semibold'><a href='#' class='unit-link' data-id='" . esc($r['unit_id']) . "'>" . esc($unit) . "</a></td>";
        echo "<td class='small-mono'>{$fecha}</td>";
        echo "<td class='small-mono'>{$hora}</td>";
        echo "<td class='small-mono'>{$coordDisp}</td>";
        echo "<td class='small-mono'>" . esc($vel) . "</td>";
        echo "<td>{$estadoBadge}</td>";
        echo "<td>{$geoBadge}</td>";
        echo "<td>{$ubic}</td>";
        echo "<td class='small-mono'>{$paro}</td>";
        echo "<td class='small-mono'>—</td>";
        echo "<td class='small-mono'>{$sinRep}</td>";
        echo "<td>{$ignCell}</td>";
        echo "</tr>";
    }

    return ob_get_clean();
}

/* =======================================================
   ✅ MODO AJAX → solo filas
   ======================================================= */
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    echo render_rows($rows, $now);
    exit;
}
?>
<!doctype html>
<html lang="es" data-theme="dark">
<head>
    <meta charset="utf-8">
    <title>Dashboard Unidades</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- Bootstrap 5 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
    <!-- SheetJS para XLSX -->
    <script src="https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js"></script>
    <!-- Leaflet (OSM) -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

    <style>
        /* ---------- Variables de tema ---------- */
        :root {
            --bg: #0b1220;
            --card: #111827;
            --text: #e7eaf0;
            --muted: #9ca3af;
            --border: #374151;
            --thead: #132038;
            --accent: #0ea5e9;
            --success: #22c55e;
            --danger: #ef4444;
            --row: #111827;
            --row-alt: #1f2937;
            --row-hover: #2d3748;
        }
        [data-theme="light"] {
            --bg: #f8fafc;
            --card: #ffffff;
            --text: #0f172a;
            --muted: #475569;
            --border: #e5e7eb;
            --thead: #e8eef9;
            --accent: #0ea5e9;
            --success: #16a34a;
            --danger: #dc2626;
            --row: #ffffff;
            --row-alt: #f8fafc;
            --row-hover: #eef2ff;
        }
        body { background: var(--bg); color: var(--text); }
        .navbar { background: linear-gradient(90deg, #0f172a, #1e293b); }
        [data-theme="light"] .navbar { background: linear-gradient(90deg, #2563eb, #60a5fa); }
        .card { background: var(--card); border: 1px solid var(--border); }
        .table thead th { background: var(--thead) !important; color: var(--text); border-bottom: 2px solid var(--border); }
        [data-theme="dark"] .table { color: var(--text); border-color: var(--border); }
        [data-theme="dark"] .table tbody tr:nth-child(even),
        [data-theme="dark"] .table tbody tr:nth-child(odd),
        [data-theme="dark"] .table tbody tr:hover { background-color: #1f2022ff }
        [data-theme="dark"] .table> :not(caption)>*>* { border-color: var(--border) !important; color: var(--text); background-color: #1f2022ff }
        .status-dot { height: 10px; width: 10px; display: inline-block; border-radius: 50%; margin-right: 6px; }
        .dot-on { background: var(--success); }
        .dot-off { background: var(--danger); }
        .dot-unk { background: #9ca3af; }
        .small-mono { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace; font-size: .85rem; color: var(--muted); }
        .badge-geo { background: var(--accent); }
        .badge-null { background: #475569; }
        [data-theme="light"] .badge-null { background: #94a3b8; }
        .btn-toggle-theme { border: 1px solid var(--border); }
        .search-input::placeholder { color: var(--muted); }
        .table-responsive { max-height: 68vh; }

        /* -------- Panel Mapa flotante -------- */
        .mapPanel {
            position: fixed; right: 24px; bottom: 24px;
            width: 460px; height: 360px; background: var(--card);
            border: 1px solid var(--border); border-radius: 12px;
            box-shadow: 0 10px 30px rgba(0,0,0,.35); display: none;
            z-index: 9999; resize: both; overflow: hidden;
        }
        .mapHeader { cursor: move; padding: 10px 12px; background: var(--thead); color: var(--text);
            display: flex; align-items: center; justify-content: space-between; border-bottom: 1px solid var(--border); }
        .mapTitle { font-weight: 600; }
        .mapClose { border: none; background: transparent; color: var(--text); }
        .mapBody { width: 100%; height: calc(100% - 44px); }
        .leaflet-container { width: 100%; height: 100%; }

        /* Marker con etiqueta flotante */
        .marker-label { display: flex; align-items: center; gap: 8px; transform: translateY(-6px); }
        .marker-dot { width: 18px; height: 18px; border-radius: 50%; border: 2px solid #fff; box-shadow: 0 0 6px rgba(0,0,0,.4); }
        .marker-text { padding: 2px 6px; border-radius: 6px; background: rgba(0,0,0,.55); color: #fff; font-weight: 600; font-size: 12px; white-space: nowrap; }
        [data-theme="light"] .marker-text { background: rgba(15,23,42,.75); }

        .unit-link { text-decoration: none; color: inherit; }
        .unit-link:hover { text-decoration: underline; }

        /* Animación de actualización de tabla */
        .fade-update { animation: fadeIn .45s ease; }
        @keyframes fadeIn { from { opacity: .35; } to { opacity: 1; } }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark mb-4">
        <div class="container-fluid">
            <span class="navbar-brand d-flex align-items-center gap-2">
                <i class="bi bi-speedometer2"></i> Dashboard Unidades
            </span>
            <div class="d-flex align-items-center ms-auto gap-3">
                <div class="small">
                    <i class="bi bi-clock-history me-1"></i>
                    <span>Actualiza tabla cada 10s</span>
                    <span class="ms-2 small-mono">Actualizado hace: <span id="agoLabel">0s</span></span>
                </div>
                <button id="btnTheme" class="btn btn-sm btn-outline-light btn-toggle-theme">
                    <i class="bi bi-moon-stars"></i> <span id="themeLabel">Oscuro</span>
                </button>
                <button id="btnCsv" class="btn btn-sm btn-outline-light"><i class="bi bi-filetype-csv me-1"></i>CSV</button>
                <button id="btnXlsx" class="btn btn-sm btn-outline-light"><i class="bi bi-file-earmark-excel me-1"></i>Excel</button>
            </div>
        </div>
    </nav>

    <div class="container-fluid">
        <div class="row g-3 mb-3">
            <div class="col-12 col-xxl-8">
                <div class="card shadow-sm">
                    <div class="card-body d-flex flex-wrap gap-3 align-items-center">
                        <div class="me-auto">
                            <div class="small text-secondary">Leyenda</div>
                            <div class="d-flex flex-wrap gap-3 mt-2">
                                <span><span class="status-dot dot-on"></span>Ignición ON</span>
                                <span><span class="status-dot dot-off"></span>Ignición OFF</span>
                                <span><span class="status-dot dot-unk"></span>Sin dato</span>
                                <span class="badge rounded-pill badge-geo">Geocerca activa</span>
                                <span class="badge rounded-pill badge-null">Sin geocerca</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- Buscador -->
            <div class="col-12 col-xxl-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <label class="form-label small text-secondary">Búsqueda dinámica (unidad, id, geocerca, estado, ubicación...)</label>
                        <input id="searchInput" type="text" class="form-control search-input" placeholder="Escribe para filtrar...">
                    </div>
                </div>
            </div>
        </div>

        <!-- Tabla -->
        <div class="card shadow-sm">
            <div class="card-body">
                <div class="table-responsive">
                    <table id="unitsTable" class="table table-striped table-bordered align-middle">
                        <thead>
                            <tr>
                                <th>Unidad</th>
                                <th>Fecha</th>
                                <th>Hora</th>
                                <th>Coordenadas</th>
                                <th>Velocidad (km/h)</th>
                                <th>Estado</th>
                                <th>Geocercas</th>
                                <th>Ubicación (dirección)</th>
                                <th>Paro de motor</th>
                                <th>Bloqueo de acelerador</th>
                                <th>Tiempo sin reportar</th>
                                <th>Tiempo de ignición</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php echo render_rows($rows, $now); ?>
                        </tbody>
                    </table>
                </div>
                <div class="d-flex justify-content-between mt-3 small text-secondary">
                    <div>Fuente: <span class="small-mono">dbs14878639.units</span></div>
                    <div>Render inicial: <span class="small-mono"><?= esc(date('Y-m-d H:i:s', $now)) ?></span></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Contenedor para paneles de mapas -->
    <div id="mapPanelsRoot"></div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

    <script>
        // ===== Utilidades comunes =====
        const htmlEl = document.documentElement;
        const agoLabel = document.getElementById('agoLabel');
        let secSinceUpdate = 0;

        // Contador "Actualizado hace ..."
        setInterval(() => {
            secSinceUpdate++;
            agoLabel.textContent = secSinceUpdate + 's';
        }, 1000);

        // Tema claro/oscuro
        (function themeToggle() {
            const btnTheme = document.getElementById('btnTheme');
            const label = document.getElementById('themeLabel');
            const savedTheme = localStorage.getItem('sti_theme') || 'dark';
            htmlEl.setAttribute('data-theme', savedTheme);
            label.textContent = savedTheme === 'dark' ? 'Oscuro' : 'Claro';
            btnTheme.addEventListener('click', () => {
                const cur = htmlEl.getAttribute('data-theme');
                const next = cur === 'dark' ? 'light' : 'dark';
                htmlEl.setAttribute('data-theme', next);
                localStorage.setItem('sti_theme', next);
                label.textContent = next === 'dark' ? 'Oscuro' : 'Claro';
            });
        })();

        // ===== Ordenar tabla con persistencia =====
        (function sortableTable() {
            const table = document.getElementById('unitsTable');
            const headers = table.querySelectorAll('th');
            const STORAGE_KEY = 'sti_sort_state';
            let sortState = JSON.parse(localStorage.getItem(STORAGE_KEY) || '{}');

            function getTbody() { return document.querySelector('#unitsTable tbody'); }

            if (sortState && sortState.col !== undefined) {
                sortTable(sortState.col, sortState.dir, true);
            }

            headers.forEach((th, idx) => {
                th.style.cursor = 'pointer';
                th.addEventListener('click', () => {
                    let dir = 'asc';
                    if (sortState.col === idx && sortState.dir === 'asc') dir = 'desc';
                    sortState = { col: idx, dir };
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(sortState));
                    sortTable(idx, dir, false);
                });
            });

            function sortTable(colIndex, dir, initialLoad) {
                const tbody = getTbody();
                const rows = Array.from(tbody.querySelectorAll('tr'));
                const multiplier = dir === 'asc' ? 1 : -1;

                rows.sort((a, b) => {
                    const aText = (a.cells[colIndex]?.innerText || '').trim().toLowerCase();
                    const bText = (b.cells[colIndex]?.innerText || '').trim().toLowerCase();

                    const aNum = parseFloat(aText.replace(',', '.'));
                    const bNum = parseFloat(bText.replace(',', '.'));
                    if (!isNaN(aNum) && !isNaN(bNum)) return (aNum - bNum) * multiplier;

                    if (/\d{4}-\d{2}-\d{2}/.test(aText) && /\d{4}-\d{2}-\d{2}/.test(bText)) {
                        return (new Date(aText) - new Date(bText)) * multiplier;
                    }
                    return aText.localeCompare(bText) * multiplier;
                });

                rows.forEach(r => tbody.appendChild(r));

                headers.forEach(h => h.classList.remove('sorted-asc', 'sorted-desc'));
                headers[colIndex].classList.add(dir === 'asc' ? 'sorted-asc' : 'sorted-desc');

                if (!initialLoad) console.log(`Tabla ordenada por columna ${colIndex} (${dir})`);
            }

            // Exponer función para re-aplicar orden tras actualización del tbody
            window.__reapplySort = function() {
                if (sortState && sortState.col !== undefined) {
                    sortTable(sortState.col, sortState.dir, true);
                }
            }
        })();

        // ===== Búsqueda dinámica =====
        (function liveSearch() {
            const searchInput = document.getElementById('searchInput');

            function getTbody() { return document.querySelector('#unitsTable tbody'); }

            function applyFilter() {
                const tbody = getTbody();
                const q = (searchInput.value || '').trim().toLowerCase();
                Array.from(tbody.rows).forEach(tr => {
                    if (!tr.dataset) return;
                    const hay = (tr.dataset.unit + ' ' + tr.dataset.id + ' ' + tr.dataset.geo + ' ' + tr.dataset.estado + ' ' + tr.dataset.ubic + ' ' + tr.innerText)
                        .toLowerCase().includes(q);
                    tr.style.display = hay ? '' : 'none';
                });
            }
            searchInput.addEventListener('input', applyFilter);
            // Exponer para re-aplicar tras actualizar tbody
            window.__reapplyFilter = applyFilter;
        })();

        // ====== SDK Mínimo de mapas (Leaflet) ======
        (function miniMapSDK() {
            const mapRoot = document.getElementById('mapPanelsRoot');
            const mapInstances = {};
            const OPEN_KEY = 'sti_open_maps';

            function getOpenMaps() {
                try { return JSON.parse(localStorage.getItem(OPEN_KEY) || '[]'); }
                catch (e) { return []; }
            }
            function setOpenMaps(arr) { localStorage.setItem(OPEN_KEY, JSON.stringify(arr)); }
            function addOpenMap(unitId) {
                const arr = getOpenMaps();
                if (!arr.includes(unitId)) { arr.push(unitId); setOpenMaps(arr); }
            }
            function removeOpenMap(unitId) {
                const arr = getOpenMaps().filter(x => x !== unitId);
                setOpenMaps(arr);
                localStorage.removeItem('sti_map_geom_' + unitId);
            }
            function parseLatLon(coordStr) {
                if (!coordStr) return null;
                const parts = coordStr.split(',');
                if (parts.length !== 2) return null;
                const lat = parseFloat(parts[0].trim());
                const lon = parseFloat(parts[1].trim());
                if (isNaN(lat) || isNaN(lon)) return null;
                return [lat, lon];
            }
            function findRowByUnitId(unitId) {
                return Array.from(document.querySelectorAll('#unitsTable tbody tr'))
                    .find(r => r.dataset && r.dataset.id === unitId);
            }
            function extractFromRow(tr) {
                const unitName = tr.dataset.unit || '';
                const ign = (tr.dataset.ign || '').toUpperCase();
                let coord = tr.dataset.coord || '';
                if (!coord || coord === '0,0') {
                    const txt = tr.cells[3]?.innerText || '';
                    if (txt && txt.includes(',')) coord = txt.trim();
                }
                const latlon = parseLatLon(coord);
                const isOn = (ign === 'ENCENDIDA' || ign === 'ON');
                return { unitName, isOn, latlon, coord };
            }
            function iconWithLabel(isOn, unitName) {
                const color = isOn ? '#22c55e' : '#ef4444';
                const html = `
                <div class="marker-label">
                    <div class="marker-dot" style="background:${color}"></div>
                    <span class="marker-text">${unitName}</span>
                </div>`;
                return L.divIcon({ className: '', html, iconSize: [1,1], iconAnchor: [0,0] });
            }
            function createPanel(unitId, unitName) {
                if (mapInstances[unitId]) return mapInstances[unitId];

                const panel = document.createElement('div');
                panel.className = 'mapPanel';
                panel.id = 'mapPanel-' + unitId;
                panel.innerHTML = `
                <div class="mapHeader">
                    <div class="mapTitle">Mapa — ${unitName}</div>
                    <div>
                        <button class="btn btn-sm btn-outline-light me-2 mapCenter" title="Centrar"><i class="bi bi-crosshair"></i></button>
                        <button class="mapClose" title="Cerrar"><i class="bi bi-x-lg"></i></button>
                    </div>
                </div>
                <div class="mapBody"><div id="map-${unitId}" class="leaflet-container"></div></div>`;
                mapRoot.appendChild(panel);

                // Restaurar geometría
                const geomKey = 'sti_map_geom_' + unitId;
                try {
                    const g = JSON.parse(localStorage.getItem(geomKey) || 'null');
                    if (g) {
                        panel.style.left = g.left; panel.style.top = g.top;
                        panel.style.right = 'auto'; panel.style.bottom = 'auto';
                        panel.style.width = g.width; panel.style.height = g.height;
                    }
                } catch (e) {}

                // Drag + persistencia
                (function makeDraggable() {
                    const header = panel.querySelector('.mapHeader');
                    let offX = 0, offY = 0, dragging = false;
                    header.addEventListener('mousedown', (e) => {
                        dragging = true;
                        const r = panel.getBoundingClientRect();
                        offX = e.clientX - r.left; offY = e.clientY - r.top;
                        document.body.style.userSelect = 'none';
                    });
                    window.addEventListener('mousemove', (e) => {
                        if (!dragging) return;
                        panel.style.left = (e.clientX - offX) + 'px';
                        panel.style.top = (e.clientY - offY) + 'px';
                        panel.style.right = 'auto'; panel.style.bottom = 'auto';
                    });
                    window.addEventListener('mouseup', () => {
                        if (dragging) {
                            dragging = false; document.body.style.userSelect = '';
                            persistGeom();
                        }
                    });
                    function persistGeom() {
                        const r = panel.getBoundingClientRect();
                        localStorage.setItem(geomKey, JSON.stringify({
                            left: panel.style.left || (window.innerWidth - r.width - 24) + 'px',
                            top: panel.style.top || (window.innerHeight - r.height - 24) + 'px',
                            width: r.width + 'px',
                            height: r.height + 'px'
                        }));
                        setTimeout(() => instance.map && instance.map.invalidateSize(), 150);
                    }
                    const ro = new ResizeObserver(() => persistGeom());
                    ro.observe(panel);
                })();

                // Mapa
                const map = L.map('map-' + unitId);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    maxZoom: 19, attribution: '&copy; OpenStreetMap'
                }).addTo(map);

                panel.querySelector('.mapClose').addEventListener('click', () => {
                    if (mapInstances[unitId]) {
                        try { mapInstances[unitId].map.remove(); } catch (e) {}
                        delete mapInstances[unitId];
                    }
                    removeOpenMap(unitId);
                    panel.remove();
                });
                panel.querySelector('.mapCenter').addEventListener('click', () => {
                    const tr = findRowByUnitId(unitId);
                    if (tr) {
                        const { latlon } = extractFromRow(tr);
                        if (latlon) { map.setView(latlon, 14); }
                    }
                });

                const instance = { panelEl: panel, map, marker: null, unitName, _hasCentered: false };
                mapInstances[unitId] = instance;

                panel.style.display = 'block';
                setTimeout(() => map.invalidateSize(), 150);
                return instance;
            }

            function openOrUpdateMapForRow(tr) {
                const unitId = tr.dataset.id;
                const { unitName, isOn, latlon } = extractFromRow(tr);
                const titleEl = () => mapInstances[unitId]?.panelEl?.querySelector('.mapTitle');

                if (!latlon) {
                    if (mapInstances[unitId]) { titleEl().textContent = `Mapa — ${unitName} (sin coordenadas)`; }
                    return;
                }
                const inst = mapInstances[unitId] || createPanel(unitId, unitName);
                addOpenMap(unitId);

                if (titleEl()) titleEl().textContent = `Mapa — ${unitName} — ${latlon[0].toFixed(6)},${latlon[1].toFixed(6)}`;

                const icon = iconWithLabel(isOn, unitName);
                if (inst.marker) { inst.map.removeLayer(inst.marker); }
                inst.marker = L.marker(latlon, { icon }).addTo(inst.map);
                if (!inst._hasCentered) { inst.map.setView(latlon, 14); inst._hasCentered = true; }
            }

            function bindRowInteractions() {
                document.querySelectorAll('.unit-link').forEach(a => {
                    a.addEventListener('click', (e) => {
                        e.preventDefault();
                        const unitId = a.dataset.id;
                        const tr = a.closest('tr');
                        if (!tr) return;
                        if (mapInstances[unitId]) {
                            const p = mapInstances[unitId].panelEl;
                            p.style.display = 'block';
                            p.style.zIndex = (parseInt(p.style.zIndex || '9999', 10) + 1).toString();
                        }
                        openOrUpdateMapForRow(tr);
                    });
                });
            }

            // Reabrir mapas que estaban abiertos
            function reopenOpenMaps() {
                const openIds = getOpenMaps();
                openIds.forEach(unitId => {
                    const tr = findRowByUnitId(unitId);
                    if (tr) openOrUpdateMapForRow(tr);
                });
            }

            // Exponer para usar tras refresh de tbody
            window.__bindRowInteractions = bindRowInteractions;
            window.__reopenOpenMaps = reopenOpenMaps;

            // En el load inicial, enlazar
            bindRowInteractions();
            reopenOpenMaps();
        })();

        // ===== Exportar CSV / XLSX =====
        (function exportsSetup() {
            document.getElementById('btnCsv').addEventListener('click', () => {
                const rows = document.querySelectorAll('#unitsTable tr');
                let csv = '';
                rows.forEach(tr => {
                    if (tr.style.display === 'none') return;
                    const cells = Array.from(tr.querySelectorAll('th,td'))
                        .map(td => '"' + (td.innerText || '').replaceAll('"', '""') + '"');
                    csv += cells.join(',') + '\n';
                });
                const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url; a.download = 'unidades.csv'; a.click();
                URL.revokeObjectURL(url);
            });

            document.getElementById('btnXlsx').addEventListener('click', () => {
                const tableElem = document.getElementById('unitsTable').cloneNode(true);
                Array.from(tableElem.tBodies[0].rows).forEach(r => { if (r.style.display === 'none') r.remove(); });
                const wb = XLSX.utils.table_to_book(tableElem, { sheet: "Unidades" });
                XLSX.writeFile(wb, 'unidades.xlsx');
            });
        })();

        // ===== PROCESO A: Ping cada 30s (NO toca la UI) =====
        (function backgroundFetchCron() {
            const FETCH_URL = "https://stilaguna.com/solusof/dashboard_data.php";
            async function ping() {
                try {
                    const resp = await fetch(FETCH_URL, { cache: "no-store" });
                    if (!resp.ok) throw new Error("HTTP " + resp.status);
                    console.log(`[${new Date().toLocaleTimeString()}] 🔄 Ping OK → ${FETCH_URL}`);
                } catch (err) {
                    console.warn("❌ Error al llamar la URL:", err.message);
                }
            }
            ping();
            setInterval(ping, 30000);
        })();

        // ===== PROCESO B: Refresco dinámico de TABLA cada 10s =====
        (function autoTableRefresh() {
            const TABLE_URL = "dashboard_units.php?ajax=1"; // este mismo archivo, modo ajax (solo filas)
            async function refreshTable() {
                try {
                    const resp = await fetch(TABLE_URL, { cache: "no-store" });
                    if (!resp.ok) throw new Error("HTTP " + resp.status);
                    const rowsHtml = await resp.text(); // solo <tr> ... </tr>
                    const tbody = document.querySelector('#unitsTable tbody');
                    if (!tbody) return;

                    // Guardar scroll para que no salte
                    const container = tbody.closest('.table-responsive');
                    const prevScroll = container ? container.scrollTop : 0;

                    tbody.innerHTML = rowsHtml;
                    tbody.classList.add('fade-update');
                    setTimeout(() => tbody.classList.remove('fade-update'), 500);

                    // Reaplicar sorting y filtro activos
                    if (window.__reapplySort) window.__reapplySort();
                    if (window.__reapplyFilter) window.__reapplyFilter();

                    // Re-enganchar eventos de filas y reabrir mapas
                    if (window.__bindRowInteractions) window.__bindRowInteractions();
                    if (window.__reopenOpenMaps) window.__reopenOpenMaps();

                    // Restaurar scroll
                    if (container) container.scrollTop = prevScroll;

                    // Reiniciar contador
                    secSinceUpdate = 0;

                    console.log(`[${new Date().toLocaleTimeString()}] ✅ Tabla actualizada dinámicamente`);
                } catch (e) {
                    console.warn("⚠️ Error actualizando tabla:", e.message);
                }
            }
            setInterval(refreshTable, 10000);
        })();
    </script>
</body>
</html>
